<?php
session_start();

// Include required files
require_once '../config/db.php';
require_once 'auth.php';

// Check if admin is logged in
requireAdminLogin();

// Get admin info
$admin = getCurrentAdmin();

$message = '';
$message_type = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db = getDB();
    
    // Handle logout
    if (isset($_POST['logout'])) {
        adminLogout();
    }
    
    // Handle add/edit slide
    if (isset($_POST['save_slide'])) {
        $slide_id = $_POST['slide_id'] ?? 0;
        $title = sanitizeInput($_POST['title'] ?? '');
        $subtitle = sanitizeInput($_POST['subtitle'] ?? '');
        $description = sanitizeInput($_POST['description'] ?? '');
        $button_text = sanitizeInput($_POST['button_text'] ?? 'Plan Your Event');
        $button_link = sanitizeInput($_POST['button_link'] ?? 'contact.php');
        $is_active = isset($_POST['is_active']) ? 1 : 0;
        $display_order = intval($_POST['display_order'] ?? 0);
        
        $image_file = '';
        
        // Handle image upload
        if (isset($_FILES['image_file']) && $_FILES['image_file']['error'] !== UPLOAD_ERR_NO_FILE) {
            $upload_dir = '../public/assets/images/hero/';
            $upload_result = handleFileUpload($_FILES['image_file'], $upload_dir, ['jpg', 'jpeg', 'png', 'webp'], 10485760); // 10MB limit
            
            if ($upload_result['success']) {
                $image_file = $upload_result['filename'];
                
                // Delete old image if editing
                if ($slide_id > 0) {
                    $stmt = $db->prepare("SELECT image_file FROM hero_slides WHERE id = ?");
                    $stmt->execute([$slide_id]);
                    $old_slide = $stmt->fetch();
                    if ($old_slide && $old_slide['image_file']) {
                        $old_file_path = $upload_dir . $old_slide['image_file'];
                        if (file_exists($old_file_path)) {
                            unlink($old_file_path);
                        }
                    }
                }
            } else {
                $message = $upload_result['message'];
                $message_type = 'error';
            }
        }
        
        if (!$message) {
            try {
                if ($slide_id > 0) {
                    // Update existing slide
                    if ($image_file) {
                        $stmt = $db->prepare("UPDATE hero_slides SET title = ?, subtitle = ?, description = ?, image_file = ?, button_text = ?, button_link = ?, is_active = ?, display_order = ? WHERE id = ?");
                        $stmt->execute([$title, $subtitle, $description, $image_file, $button_text, $button_link, $is_active, $display_order, $slide_id]);
                    } else {
                        $stmt = $db->prepare("UPDATE hero_slides SET title = ?, subtitle = ?, description = ?, button_text = ?, button_link = ?, is_active = ?, display_order = ? WHERE id = ?");
                        $stmt->execute([$title, $subtitle, $description, $button_text, $button_link, $is_active, $display_order, $slide_id]);
                    }
                    $message = 'Hero slide updated successfully!';
                } else {
                    // Add new slide
                    if (!$image_file) {
                        $message = 'Please select an image for the hero slide.';
                        $message_type = 'error';
                    } else {
                        $stmt = $db->prepare("INSERT INTO hero_slides (title, subtitle, description, image_file, button_text, button_link, is_active, display_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                        $stmt->execute([$title, $subtitle, $description, $image_file, $button_text, $button_link, $is_active, $display_order]);
                        $message = 'Hero slide added successfully!';
                    }
                }
                
                if (!$message_type) {
                    $message_type = 'success';
                }
            } catch (Exception $e) {
                $message = 'Database error: ' . $e->getMessage();
                $message_type = 'error';
            }
        }
    }
    
    // Handle delete slide
    if (isset($_POST['delete_slide'])) {
        $slide_id = intval($_POST['slide_id']);
        
        try {
            // Get image file to delete
            $stmt = $db->prepare("SELECT image_file FROM hero_slides WHERE id = ?");
            $stmt->execute([$slide_id]);
            $slide = $stmt->fetch();
            
            if ($slide) {
                // Delete database record
                $stmt = $db->prepare("DELETE FROM hero_slides WHERE id = ?");
                $stmt->execute([$slide_id]);
                
                // Delete image file
                $image_path = '../public/assets/images/hero/' . $slide['image_file'];
                if (file_exists($image_path)) {
                    unlink($image_path);
                }
                
                $message = 'Hero slide deleted successfully!';
                $message_type = 'success';
            }
        } catch (Exception $e) {
            $message = 'Error deleting slide: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    // Handle toggle active status
    if (isset($_POST['toggle_active'])) {
        $slide_id = intval($_POST['slide_id']);
        $is_active = intval($_POST['is_active']);
        
        try {
            $stmt = $db->prepare("UPDATE hero_slides SET is_active = ? WHERE id = ?");
            $stmt->execute([$is_active, $slide_id]);
            $message = 'Slide status updated successfully!';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Error updating slide status: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Get all hero slides
try {
    $db = getDB();
    $stmt = $db->query("SELECT * FROM hero_slides ORDER BY display_order, created_at DESC");
    $slides = $stmt->fetchAll();
} catch (Exception $e) {
    $slides = [];
    if (!$message) {
        $message = 'Error loading slides: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Get slide for editing
$edit_slide = null;
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $edit_id = intval($_GET['edit']);
    foreach ($slides as $slide) {
        if ($slide['id'] == $edit_id) {
            $edit_slide = $slide;
            break;
        }
    }
}

$page_title = 'Hero Slider Management';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Admin Panel</title>
    <link rel="icon" type="image/x-icon" href="../public/favicon.ico">
    
    <!-- Google Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            --primary-purple: #6B46C1;
            --primary-gold: #B8860B;
            --primary-maroon: #8B1538;
            --text-primary: #1F2937;
            --text-secondary: #6B7280;
            --bg-light: #F9FAFB;
            --bg-white: #FFFFFF;
            --border-color: #E5E7EB;
            --success-green: #10B981;
            --warning-yellow: #F59E0B;
            --error-red: #EF4444;
            --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
            --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
            --radius-md: 8px;
            --radius-lg: 12px;
            --transition: all 0.3s ease;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: var(--bg-light);
            color: var(--text-primary);
            line-height: 1.6;
        }
        
        .admin-layout {
            display: flex;
            min-height: 100vh;
        }
        
        /* Sidebar - Same as dashboard */
        .sidebar {
            width: 280px;
            background: var(--bg-white);
            border-right: 1px solid var(--border-color);
            padding: 2rem 0;
            position: fixed;
            height: 100vh;
            overflow-y: auto;
            box-shadow: var(--shadow-sm);
        }
        
        .sidebar-header {
            padding: 0 2rem 2rem;
            border-bottom: 1px solid var(--border-color);
            margin-bottom: 2rem;
        }
        
        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .sidebar-logo .logo-icon {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .sidebar-title {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .sidebar-subtitle {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .sidebar-nav {
            padding: 0 1rem;
        }
        
        .nav-item {
            margin-bottom: 0.5rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            color: var(--text-secondary);
            text-decoration: none;
            border-radius: var(--radius-md);
            transition: var(--transition);
            font-weight: 500;
        }
        
        .nav-link:hover {
            background: var(--bg-light);
            color: var(--primary-purple);
        }
        
        .nav-link.active {
            background: var(--primary-purple);
            color: white;
        }
        
        .nav-link i {
            width: 20px;
            text-align: center;
        }
        
        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 280px;
            padding: 2rem;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .admin-info {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .admin-avatar {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
        }
        
        .admin-details {
            display: flex;
            flex-direction: column;
        }
        
        .admin-name {
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .admin-role {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .logout-btn {
            background: var(--error-red);
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: var(--radius-md);
            cursor: pointer;
            font-size: 0.85rem;
            transition: var(--transition);
        }
        
        .logout-btn:hover {
            background: #DC2626;
        }
        
        /* Alert Messages */
        .alert {
            padding: 1rem 1.5rem;
            border-radius: var(--radius-md);
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            border: 1px solid rgba(16, 185, 129, 0.2);
            color: var(--success-green);
        }
        
        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid rgba(239, 68, 68, 0.2);
            color: var(--error-red);
        }
        
        /* Form Styles */
        .form-section {
            background: var(--bg-white);
            padding: 2rem;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            margin-bottom: 2rem;
        }
        
        .section-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 1.5rem;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid var(--primary-gold);
        }
        
        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1.5rem;
            margin-bottom: 1.5rem;
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-group.full-width {
            grid-column: 1 / -1;
        }
        
        .form-label {
            display: block;
            font-weight: 500;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }
        
        .form-input {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            font-size: 1rem;
            transition: var(--transition);
        }
        
        .form-input:focus {
            outline: none;
            border-color: var(--primary-purple);
            box-shadow: 0 0 0 3px rgba(107, 70, 193, 0.1);
        }
        
        .form-textarea {
            min-height: 100px;
            resize: vertical;
        }
        
        .form-checkbox {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 1rem;
        }
        
        .form-checkbox input[type="checkbox"] {
            width: 18px;
            height: 18px;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: var(--radius-md);
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .btn-primary {
            background: var(--primary-purple);
            color: white;
        }
        
        .btn-primary:hover {
            background: #553C9A;
        }
        
        .btn-secondary {
            background: var(--text-secondary);
            color: white;
        }
        
        .btn-secondary:hover {
            background: #4B5563;
        }
        
        .btn-danger {
            background: var(--error-red);
            color: white;
        }
        
        .btn-danger:hover {
            background: #DC2626;
        }
        
        /* Slides Grid */
        .slides-section {
            background: var(--bg-white);
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            overflow: hidden;
        }
        
        .slides-header {
            padding: 1.5rem 2rem;
            background: var(--bg-light);
            border-bottom: 1px solid var(--border-color);
        }
        
        .slides-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 1.5rem;
            padding: 2rem;
        }
        
        .slide-card {
            border: 1px solid var(--border-color);
            border-radius: var(--radius-lg);
            overflow: hidden;
            transition: var(--transition);
        }
        
        .slide-card:hover {
            box-shadow: var(--shadow-lg);
        }
        
        .slide-image {
            width: 100%;
            height: 200px;
            object-fit: cover;
            background: var(--bg-light);
        }
        
        .slide-content {
            padding: 1.5rem;
        }
        
        .slide-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }
        
        .slide-subtitle {
            font-size: 0.9rem;
            color: var(--primary-gold);
            margin-bottom: 1rem;
            font-style: italic;
        }
        
        .slide-description {
            font-size: 0.85rem;
            color: var(--text-secondary);
            line-height: 1.6;
            margin-bottom: 1rem;
        }
        
        .slide-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .slide-actions {
            display: flex;
            gap: 0.5rem;
        }
        
        .btn-sm {
            padding: 0.5rem 0.75rem;
            font-size: 0.8rem;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .status-active {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success-green);
        }
        
        .status-inactive {
            background: rgba(107, 114, 128, 0.1);
            color: var(--text-secondary);
        }
        
        .empty-state {
            padding: 3rem;
            text-align: center;
            color: var(--text-secondary);
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: var(--text-secondary);
            opacity: 0.5;
        }
        
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                height: auto;
                position: relative;
            }
            
            .main-content {
                margin-left: 0;
                padding: 1rem;
            }
            
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .slides-grid {
                grid-template-columns: 1fr;
                padding: 1rem;
            }
            
            .page-header {
                flex-direction: column;
                gap: 1rem;
                align-items: flex-start;
            }
            
            .admin-info {
                width: 100%;
                justify-content: space-between;
            }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">
                    <div class="logo-icon">
                        <i class="fas fa-crown"></i>
                    </div>
                    <div>
                        <div class="sidebar-title">SSV Admin</div>
                        <div class="sidebar-subtitle">Event Management</div>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <div class="nav-item">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-tachometer-alt"></i>
                        Dashboard
                    </a>
                </div>
                <div class="nav-item">
                    <a href="hero-slides.php" class="nav-link active">
                        <i class="fas fa-images"></i>
                        Hero Slider
                    </a>
                </div>
                <div class="nav-item">
                    <a href="services.php" class="nav-link">
                        <i class="fas fa-concierge-bell"></i>
                        Services
                    </a>
                </div>
                <div class="nav-item">
                    <a href="gallery.php" class="nav-link">
                        <i class="fas fa-photo-video"></i>
                        Gallery
                    </a>
                </div>
                <div class="nav-item">
                    <a href="testimonials.php" class="nav-link">
                        <i class="fas fa-star"></i>
                        Testimonials
                    </a>
                </div>
                <div class="nav-item">
                    <a href="enquiries.php" class="nav-link">
                        <i class="fas fa-envelope"></i>
                        Enquiries
                    </a>
                </div>
                <div class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        Settings
                    </a>
                </div>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="page-header">
                <h1 class="page-title"><?php echo $page_title; ?></h1>
                <div class="admin-info">
                    <div class="admin-avatar">
                        <?php echo strtoupper(substr($admin['name'], 0, 1)); ?>
                    </div>
                    <div class="admin-details">
                        <div class="admin-name"><?php echo htmlspecialchars($admin['name']); ?></div>
                        <div class="admin-role">Administrator</div>
                    </div>
                    <form method="POST" style="margin-left: 1rem;">
                        <button type="submit" name="logout" class="logout-btn">
                            <i class="fas fa-sign-out-alt"></i> Logout
                        </button>
                    </form>
                </div>
            </div>
            
            <!-- Alert Messages -->
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?>">
                    <i class="fas fa-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <!-- Add/Edit Slide Form -->
            <div class="form-section">
                <h2 class="section-title">
                    <?php echo $edit_slide ? 'Edit Hero Slide' : 'Add New Hero Slide'; ?>
                </h2>
                
                <form method="POST" enctype="multipart/form-data">
                    <?php if ($edit_slide): ?>
                        <input type="hidden" name="slide_id" value="<?php echo $edit_slide['id']; ?>">
                    <?php endif; ?>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="title" class="form-label">Title *</label>
                            <input type="text" id="title" name="title" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_slide['title'] ?? ''); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="subtitle" class="form-label">Subtitle</label>
                            <input type="text" id="subtitle" name="subtitle" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_slide['subtitle'] ?? ''); ?>">
                        </div>
                        
                        <div class="form-group full-width">
                            <label for="description" class="form-label">Description</label>
                            <textarea id="description" name="description" class="form-input form-textarea"><?php echo htmlspecialchars($edit_slide['description'] ?? ''); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="image_file" class="form-label">Hero Image <?php echo $edit_slide ? '' : '*'; ?></label>
                            <input type="file" id="image_file" name="image_file" class="form-input" 
                                   accept="image/*" <?php echo $edit_slide ? '' : 'required'; ?>>
                            <?php if ($edit_slide && $edit_slide['image_file']): ?>
                                <small style="color: var(--text-secondary); margin-top: 0.5rem; display: block;">
                                    Current image: <?php echo htmlspecialchars($edit_slide['image_file']); ?>
                                </small>
                            <?php endif; ?>
                        </div>
                        
                        <div class="form-group">
                            <label for="display_order" class="form-label">Display Order</label>
                            <input type="number" id="display_order" name="display_order" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_slide['display_order'] ?? '0'); ?>" min="0">
                        </div>
                        
                        <div class="form-group">
                            <label for="button_text" class="form-label">Button Text</label>
                            <input type="text" id="button_text" name="button_text" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_slide['button_text'] ?? 'Plan Your Event'); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="button_link" class="form-label">Button Link</label>
                            <input type="text" id="button_link" name="button_link" class="form-input" 
                                   value="<?php echo htmlspecialchars($edit_slide['button_link'] ?? 'contact.php'); ?>">
                        </div>
                    </div>
                    
                    <div class="form-checkbox">
                        <input type="checkbox" id="is_active" name="is_active" <?php echo ($edit_slide['is_active'] ?? 1) ? 'checked' : ''; ?>>
                        <label for="is_active" class="form-label" style="margin-bottom: 0;">Active</label>
                    </div>
                    
                    <div style="display: flex; gap: 1rem;">
                        <button type="submit" name="save_slide" class="btn btn-primary">
                            <i class="fas fa-save"></i>
                            <?php echo $edit_slide ? 'Update Slide' : 'Add Slide'; ?>
                        </button>
                        
                        <?php if ($edit_slide): ?>
                            <a href="hero-slides.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i>
                                Cancel
                            </a>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
            
            <!-- Existing Slides -->
            <div class="slides-section">
                <div class="slides-header">
                    <h2 class="section-title">Existing Hero Slides</h2>
                </div>
                
                <?php if (empty($slides)): ?>
                    <div class="empty-state">
                        <i class="fas fa-images"></i>
                        <h3>No hero slides yet</h3>
                        <p>Add your first hero slide using the form above.</p>
                    </div>
                <?php else: ?>
                    <div class="slides-grid">
                        <?php foreach ($slides as $slide): ?>
                            <div class="slide-card">
                                <?php if ($slide['image_file'] && file_exists('../public/assets/images/hero/' . $slide['image_file'])): ?>
                                    <img src="../public/assets/images/hero/<?php echo htmlspecialchars($slide['image_file']); ?>" 
                                         alt="<?php echo htmlspecialchars($slide['title']); ?>" class="slide-image">
                                <?php else: ?>
                                    <div class="slide-image" style="display: flex; align-items: center; justify-content: center; background: var(--bg-light); color: var(--text-secondary);">
                                        <i class="fas fa-image" style="font-size: 2rem;"></i>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="slide-content">
                                    <h3 class="slide-title"><?php echo htmlspecialchars($slide['title']); ?></h3>
                                    
                                    <?php if ($slide['subtitle']): ?>
                                        <p class="slide-subtitle"><?php echo htmlspecialchars($slide['subtitle']); ?></p>
                                    <?php endif; ?>
                                    
                                    <?php if ($slide['description']): ?>
                                        <p class="slide-description"><?php echo htmlspecialchars(substr($slide['description'], 0, 100)) . (strlen($slide['description']) > 100 ? '...' : ''); ?></p>
                                    <?php endif; ?>
                                    
                                    <div class="slide-meta">
                                        <div>
                                            <span class="status-badge status-<?php echo $slide['is_active'] ? 'active' : 'inactive'; ?>">
                                                <?php echo $slide['is_active'] ? 'Active' : 'Inactive'; ?>
                                            </span>
                                        </div>
                                        <div>Order: <?php echo $slide['display_order']; ?></div>
                                    </div>
                                    
                                    <div class="slide-actions">
                                        <a href="hero-slides.php?edit=<?php echo $slide['id']; ?>" class="btn btn-secondary btn-sm">
                                            <i class="fas fa-edit"></i> Edit
                                        </a>
                                        
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this slide?');">
                                            <input type="hidden" name="slide_id" value="<?php echo $slide['id']; ?>">
                                            <button type="submit" name="delete_slide" class="btn btn-danger btn-sm">
                                                <i class="fas fa-trash"></i> Delete
                                            </button>
                                        </form>
                                        
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="slide_id" value="<?php echo $slide['id']; ?>">
                                            <input type="hidden" name="is_active" value="<?php echo $slide['is_active'] ? 0 : 1; ?>">
                                            <button type="submit" name="toggle_active" class="btn btn-<?php echo $slide['is_active'] ? 'secondary' : 'primary'; ?> btn-sm">
                                                <i class="fas fa-<?php echo $slide['is_active'] ? 'eye-slash' : 'eye'; ?>"></i>
                                                <?php echo $slide['is_active'] ? 'Hide' : 'Show'; ?>
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>