<?php
session_start();

require_once '../config/db.php';
require_once 'auth.php';
requireAdminLogin();

$admin = getCurrentAdmin();
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db = getDB();
    
    if (isset($_POST['logout'])) {
        adminLogout();
    }
    
    if (isset($_POST['save_testimonial'])) {
        $testimonial_id = $_POST['testimonial_id'] ?? 0;
        $name = sanitizeInput($_POST['name'] ?? '');
        $email = sanitizeInput($_POST['email'] ?? '');
        $phone = sanitizeInput($_POST['phone'] ?? '');
        $event_type = sanitizeInput($_POST['event_type'] ?? '');
        $event_date = sanitizeInput($_POST['event_date'] ?? '');
        $rating = intval($_POST['rating'] ?? 5);
        $testimonial_text = sanitizeInput($_POST['testimonial_text'] ?? '');
        $status = sanitizeInput($_POST['status'] ?? 'pending');
        $show_on_home = isset($_POST['show_on_home']) ? 1 : 0;
        $admin_notes = sanitizeInput($_POST['admin_notes'] ?? '');
        
        try {
            if ($testimonial_id > 0) {
                $stmt = $db->prepare("UPDATE testimonials SET name = ?, email = ?, phone = ?, event_type = ?, event_date = ?, rating = ?, testimonial = ?, status = ?, show_on_home = ?, admin_notes = ? WHERE id = ?");
                $stmt->execute([$name, $email, $phone, $event_type, $event_date, $rating, $testimonial_text, $status, $show_on_home, $admin_notes, $testimonial_id]);
                $message = 'Testimonial updated successfully!';
            } else {
                $stmt = $db->prepare("INSERT INTO testimonials (name, email, phone, event_type, event_date, rating, testimonial, status, show_on_home, admin_notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$name, $email, $phone, $event_type, $event_date, $rating, $testimonial_text, $status, $show_on_home, $admin_notes]);
                $message = 'Testimonial added successfully!';
            }
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Database error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['delete_testimonial'])) {
        try {
            $stmt = $db->prepare("DELETE FROM testimonials WHERE id = ?");
            $stmt->execute([intval($_POST['testimonial_id'])]);
            $message = 'Testimonial deleted successfully!';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['update_status'])) {
        try {
            $stmt = $db->prepare("UPDATE testimonials SET status = ? WHERE id = ?");
            $stmt->execute([sanitizeInput($_POST['new_status']), intval($_POST['testimonial_id'])]);
            $message = 'Status updated successfully!';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
    
    if (isset($_POST['toggle_home_display'])) {
        try {
            $stmt = $db->prepare("UPDATE testimonials SET show_on_home = ? WHERE id = ?");
            $stmt->execute([intval($_POST['value']), intval($_POST['testimonial_id'])]);
            $message = 'Home display updated!';
            $message_type = 'success';
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Get testimonials
try {
    $db = getDB();
    $filter = $_GET['filter'] ?? 'all';
    $search = $_GET['search'] ?? '';
    
    $query = "SELECT * FROM testimonials WHERE 1=1";
    $params = [];
    
    if ($filter !== 'all') {
        $query .= " AND status = ?";
        $params[] = $filter;
    }
    
    if (!empty($search)) {
        $query .= " AND (name LIKE ? OR email LIKE ? OR testimonial LIKE ?)";
        $search_term = '%' . $search . '%';
        $params[] = $search_term;
        $params[] = $search_term;
        $params[] = $search_term;
    }
    
    $query .= " ORDER BY created_at DESC";
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $testimonials = $stmt->fetchAll();
    
    // Get counts for filters
    $stmt = $db->query("SELECT status, COUNT(*) as count FROM testimonials GROUP BY status");
    $status_counts = [];
    while ($row = $stmt->fetch()) {
        $status_counts[$row['status']] = $row['count'];
    }
    $status_counts['all'] = array_sum($status_counts);
    
} catch (Exception $e) {
    $testimonials = [];
    $status_counts = ['all' => 0];
}

$event_types = ['Wedding', 'Engagement', 'Birthday', 'Corporate', 'Reception', 'Anniversary', 'Other'];
$statuses = ['pending', 'approved', 'rejected'];

$edit_testimonial = null;
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    foreach ($testimonials as $testimonial) {
        if ($testimonial['id'] == intval($_GET['edit'])) {
            $edit_testimonial = $testimonial;
            break;
        }
    }
}

$page_title = 'Testimonials Management';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Admin Panel</title>
    <link rel="icon" type="image/x-icon" href="../public/favicon.ico">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-purple: #6B46C1;
            --primary-gold: #B8860B;
            --text-primary: #1F2937;
            --text-secondary: #6B7280;
            --bg-light: #F9FAFB;
            --bg-white: #FFFFFF;
            --border-color: #E5E7EB;
            --success-green: #10B981;
            --warning-yellow: #F59E0B;
            --error-red: #EF4444;
            --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            --radius-md: 8px;
            --radius-lg: 12px;
            --transition: all 0.3s ease;
        }
        
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: var(--bg-light);
            color: var(--text-primary);
            line-height: 1.6;
        }
        
        .admin-layout { display: flex; min-height: 100vh; }
        
        .sidebar {
            width: 280px;
            background: var(--bg-white);
            border-right: 1px solid var(--border-color);
            padding: 2rem 0;
            position: fixed;
            height: 100vh;
            overflow-y: auto;
            box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
        }
        
        .sidebar-header {
            padding: 0 2rem 2rem;
            border-bottom: 1px solid var(--border-color);
            margin-bottom: 2rem;
        }
        
        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .sidebar-logo .logo-icon {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .sidebar-title {
            font-size: 1.2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .sidebar-subtitle {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .sidebar-nav { padding: 0 1rem; }
        
        .nav-item { margin-bottom: 0.5rem; }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            color: var(--text-secondary);
            text-decoration: none;
            border-radius: var(--radius-md);
            transition: var(--transition);
            font-weight: 500;
        }
        
        .nav-link:hover {
            background: var(--bg-light);
            color: var(--primary-purple);
        }
        
        .nav-link.active {
            background: var(--primary-purple);
            color: white;
        }
        
        .nav-link i { width: 20px; text-align: center; }
        
        .main-content {
            flex: 1;
            margin-left: 280px;
            padding: 2rem;
        }
        
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .page-title {
            font-size: 2rem;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        .admin-info {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .admin-avatar {
            width: 40px;
            height: 40px;
            background: var(--primary-gold);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
        }
        
        .admin-details {
            display: flex;
            flex-direction: column;
        }
        
        .admin-name {
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .admin-role {
            font-size: 0.8rem;
            color: var(--text-secondary);
        }
        
        .logout-btn {
            background: var(--error-red);
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: var(--radius-md);
            cursor: pointer;
            font-size: 0.85rem;
            transition: var(--transition);
        }
        
        .logout-btn:hover { background: #DC2626; }
        
        .alert {
            padding: 1rem 1.5rem;
            border-radius: var(--radius-md);
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            border: 1px solid rgba(16, 185, 129, 0.2);
            color: var(--success-green);
        }
        
        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid rgba(239, 68, 68, 0.2);
            color: var(--error-red);
        }
        
        .form-section {
            background: var(--bg-white);
            padding: 2rem;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-md);
            margin-bottom: 2rem;
        }
        
        .section-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 1.5rem;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid var(--primary-gold);
        }
        
        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1.5rem;
            margin-bottom: 1.5rem;
        }
        
        .form-group { margin-bottom: 1.5rem; }
        
        .form-group.full-width { grid-column: 1 / -1; }
        
        .form-label {
            display: block;
            font-weight: 500;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }
        
        .form-input {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            font-size: 1rem;
            transition: var(--transition);
        }
        
        .form-input:focus {
            outline: none;
            border-color: var(--primary-purple);
            box-shadow: 0 0 0 3px rgba(107, 70, 193, 0.1);
        }
        
        .form-textarea { min-height: 120px; resize: vertical; }
        
        .form-checkbox {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-bottom: 1rem;
        }
        
        .form-checkbox input[type="checkbox"] {
            width: 18px;
            height: 18px;
        }
        
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: var(--radius-md);
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .btn-sm { padding: 0.5rem 0.75rem; font-size: 0.8rem; }
        .btn-primary { background: var(--primary-purple); color: white; }
        .btn-primary:hover { background: #553C9A; }
        .btn-secondary { background: var(--text-secondary); color: white; }
        .btn-secondary:hover { background: #4B5563; }
        .btn-danger { background: var(--error-red); color: white; }
        .btn-danger:hover { background: #DC2626; }
        .btn-success { background: var(--success-green); color: white; }
        .btn-success:hover { background: #059669; }
        .btn-warning { background: var(--warning-yellow); color: white; }
        .btn-warning:hover { background: #D97706; }
        
        .filters-section {
            display: flex;
            gap: 1rem;
            margin-bottom: 2rem;
            flex-wrap: wrap;
        }
        
        .filter-tabs {
            display: flex;
            gap: 0.5rem;
        }
        
        .filter-tab {
            padding: 0.5rem 1rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            background: var(--bg-white);
            color: var(--text-secondary);
            text-decoration: none;
            font-size: 0.9rem;
            transition: var(--transition);
        }
        
        .filter-tab.active {
            background: var(--primary-purple);
            color: white;
            border-color: var(--primary-purple);
        }
        
        .search-form {
            display: flex;
            gap: 0.5rem;
        }
        
        .search-input {
            padding: 0.5rem 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: var(--radius-md);
            font-size: 0.9rem;
            width: 250px;
        }
        
        .testimonial-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 1.5rem;
        }
        
        .testimonial-card {
            border: 1px solid var(--border-color);
            border-radius: var(--radius-lg);
            padding: 1.5rem;
            background: var(--bg-white);
            transition: var(--transition);
        }
        
        .testimonial-card:hover { box-shadow: var(--shadow-md); }
        
        .testimonial-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }
        
        .customer-info h3 {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.25rem;
        }
        
        .customer-meta {
            font-size: 0.85rem;
            color: var(--text-secondary);
            margin-bottom: 0.5rem;
        }
        
        .rating-display {
            display: flex;
            align-items: center;
            gap: 0.25rem;
        }
        
        .star {
            color: var(--warning-yellow);
            font-size: 0.9rem;
        }
        
        .star.empty {
            color: var(--border-color);
        }
        
        .testimonial-text {
            font-style: italic;
            color: var(--text-primary);
            line-height: 1.6;
            margin: 1rem 0;
        }
        
        .testimonial-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 1.5rem;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.7rem;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .status-pending {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning-yellow);
        }
        
        .status-approved {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success-green);
        }
        
        .status-rejected {
            background: rgba(239, 68, 68, 0.1);
            color: var(--error-red);
        }
        
        .home-display-badge {
            background: rgba(107, 70, 193, 0.1);
            color: var(--primary-purple);
        }
        
        .testimonial-actions {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }
        
        .admin-notes {
            background: var(--bg-light);
            padding: 1rem;
            border-radius: var(--radius-md);
            margin: 1rem 0;
            font-size: 0.9rem;
        }
        
        .empty-state {
            padding: 3rem;
            text-align: center;
            color: var(--text-secondary);
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }
        
        @media (max-width: 768px) {
            .sidebar { width: 100%; height: auto; position: relative; }
            .main-content { margin-left: 0; padding: 1rem; }
            .form-grid { grid-template-columns: 1fr; }
            .page-header { flex-direction: column; gap: 1rem; align-items: flex-start; }
            .filters-section { flex-direction: column; }
            .search-input { width: 100%; }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">
                    <div class="logo-icon">
                        <i class="fas fa-crown"></i>
                    </div>
                    <div>
                        <div class="sidebar-title">SSV Admin</div>
                        <div class="sidebar-subtitle">Event Management</div>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <div class="nav-item">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-tachometer-alt"></i>
                        Dashboard
                    </a>
                </div>
                <div class="nav-item">
                    <a href="hero-slides.php" class="nav-link">
                        <i class="fas fa-images"></i>
                        Hero Slider
                    </a>
                </div>
                <div class="nav-item">
                    <a href="services.php" class="nav-link">
                        <i class="fas fa-concierge-bell"></i>
                        Services
                    </a>
                </div>
                <div class="nav-item">
                    <a href="gallery.php" class="nav-link">
                        <i class="fas fa-photo-video"></i>
                        Gallery
                    </a>
                </div>
                <div class="nav-item">
                    <a href="testimonials.php" class="nav-link active">
                        <i class="fas fa-star"></i>
                        Testimonials
                    </a>
                </div>
                <div class="nav-item">
                    <a href="enquiries.php" class="nav-link">
                        <i class="fas fa-envelope"></i>
                        Enquiries
                    </a>
                </div>
                <div class="nav-item">
                    <a href="settings.php" class="nav-link">
                        <i class="fas fa-cog"></i>
                        Settings
                    </a>
                </div>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="page-header">
                <h1 class="page-title"><?php echo $page_title; ?></h1>
                <div class="admin-info">
                    <div class="admin-avatar">
                        <?php echo strtoupper(substr($admin['name'], 0, 1)); ?>
                    </div>
                    <div class="admin-details">
                        <div class="admin-name"><?php echo htmlspecialchars($admin['name']); ?></div>
                        <div class="admin-role">Administrator</div>
                    </div>
                    <form method="POST" style="margin-left: 1rem;">
                        <button type="submit" name="logout" class="logout-btn">
                            <i class="fas fa-sign-out-alt"></i> Logout
                        </button>
                    </form>
                </div>
            </div>
            
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type; ?>">
                    <i class="fas fa-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <!-- Add/Edit Form -->
            <?php if ($edit_testimonial || isset($_GET['add'])): ?>
            <div class="form-section">
                <h2 class="section-title">
                    <?php echo $edit_testimonial ? 'Edit Testimonial' : 'Add New Testimonial'; ?>
                </h2>
                
                <form method="POST">
                    <?php if ($edit_testimonial): ?>
                        <input type="hidden" name="testimonial_id" value="<?php echo $edit_testimonial['id']; ?>">
                    <?php endif; ?>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="name" class="form-label">Customer Name *</label>
                            <input type="text" id="name" name="name" class="form-input" required
                                   value="<?php echo htmlspecialchars($edit_testimonial['name'] ?? ''); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="email" class="form-label">Email</label>
                            <input type="email" id="email" name="email" class="form-input"
                                   value="<?php echo htmlspecialchars($edit_testimonial['email'] ?? ''); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="phone" class="form-label">Phone</label>
                            <input type="tel" id="phone" name="phone" class="form-input"
                                   value="<?php echo htmlspecialchars($edit_testimonial['phone'] ?? ''); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="event_type" class="form-label">Event Type</label>
                            <select id="event_type" name="event_type" class="form-input">
                                <option value="">Select Event Type</option>
                                <?php foreach ($event_types as $type): ?>
                                    <option value="<?php echo $type; ?>" <?php echo ($edit_testimonial['event_type'] ?? '') === $type ? 'selected' : ''; ?>>
                                        <?php echo $type; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="event_date" class="form-label">Event Date</label>
                            <input type="date" id="event_date" name="event_date" class="form-input"
                                   value="<?php echo htmlspecialchars($edit_testimonial['event_date'] ?? ''); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="rating" class="form-label">Rating</label>
                            <select id="rating" name="rating" class="form-input">
                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                    <option value="<?php echo $i; ?>" <?php echo ($edit_testimonial['rating'] ?? 5) == $i ? 'selected' : ''; ?>>
                                        <?php echo $i; ?> Star<?php echo $i > 1 ? 's' : ''; ?>
                                    </option>
                                <?php endfor; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="status" class="form-label">Status</label>
                            <select id="status" name="status" class="form-input">
                                <?php foreach ($statuses as $status): ?>
                                    <option value="<?php echo $status; ?>" <?php echo ($edit_testimonial['status'] ?? 'pending') === $status ? 'selected' : ''; ?>>
                                        <?php echo ucfirst($status); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group full-width">
                            <label for="testimonial_text" class="form-label">Testimonial Text *</label>
                            <textarea id="testimonial_text" name="testimonial_text" class="form-input form-textarea" required><?php echo htmlspecialchars($edit_testimonial['testimonial'] ?? ''); ?></textarea>
                        </div>
                        
                        <div class="form-group full-width">
                            <label for="admin_notes" class="form-label">Admin Notes (Internal)</label>
                            <textarea id="admin_notes" name="admin_notes" class="form-input" placeholder="Internal notes for admin use only..."><?php echo htmlspecialchars($edit_testimonial['admin_notes'] ?? ''); ?></textarea>
                        </div>
                    </div>
                    
                    <div class="form-checkbox">
                        <input type="checkbox" id="show_on_home" name="show_on_home" <?php echo ($edit_testimonial['show_on_home'] ?? 0) ? 'checked' : ''; ?>>
                        <label for="show_on_home" class="form-label" style="margin-bottom: 0;">Show on Home Page</label>
                    </div>
                    
                    <div style="display: flex; gap: 1rem;">
                        <button type="submit" name="save_testimonial" class="btn btn-primary">
                            <i class="fas fa-save"></i>
                            <?php echo $edit_testimonial ? 'Update Testimonial' : 'Add Testimonial'; ?>
                        </button>
                        
                        <a href="testimonials.php" class="btn btn-secondary">
                            <i class="fas fa-times"></i>
                            Cancel
                        </a>
                    </div>
                </form>
            </div>
            <?php endif; ?>
            
            <!-- Filters and Search -->
            <div class="filters-section">
                <div class="filter-tabs">
                    <a href="testimonials.php?filter=all<?php echo !empty($_GET['search']) ? '&search=' . urlencode($_GET['search']) : ''; ?>" 
                       class="filter-tab <?php echo ($filter === 'all') ? 'active' : ''; ?>">
                        All (<?php echo $status_counts['all'] ?? 0; ?>)
                    </a>
                    <a href="testimonials.php?filter=pending<?php echo !empty($_GET['search']) ? '&search=' . urlencode($_GET['search']) : ''; ?>" 
                       class="filter-tab <?php echo ($filter === 'pending') ? 'active' : ''; ?>">
                        Pending (<?php echo $status_counts['pending'] ?? 0; ?>)
                    </a>
                    <a href="testimonials.php?filter=approved<?php echo !empty($_GET['search']) ? '&search=' . urlencode($_GET['search']) : ''; ?>" 
                       class="filter-tab <?php echo ($filter === 'approved') ? 'active' : ''; ?>">
                        Approved (<?php echo $status_counts['approved'] ?? 0; ?>)
                    </a>
                    <a href="testimonials.php?filter=rejected<?php echo !empty($_GET['search']) ? '&search=' . urlencode($_GET['search']) : ''; ?>" 
                       class="filter-tab <?php echo ($filter === 'rejected') ? 'active' : ''; ?>">
                        Rejected (<?php echo $status_counts['rejected'] ?? 0; ?>)
                    </a>
                </div>
                
                <form method="GET" class="search-form">
                    <input type="hidden" name="filter" value="<?php echo htmlspecialchars($filter); ?>">
                    <input type="text" name="search" class="search-input" placeholder="Search testimonials..." 
                           value="<?php echo htmlspecialchars($search); ?>">
                    <button type="submit" class="btn btn-secondary">
                        <i class="fas fa-search"></i> Search
                    </button>
                </form>
                
                <a href="testimonials.php?add=1" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Add Testimonial
                </a>
            </div>
            
            <!-- Testimonials List -->
            <div class="form-section">
                <h2 class="section-title">Testimonials</h2>
                
                <?php if (empty($testimonials)): ?>
                    <div class="empty-state">
                        <i class="fas fa-star"></i>
                        <h3>No testimonials found</h3>
                        <p>No testimonials match your current filters.</p>
                    </div>
                <?php else: ?>
                    <div class="testimonial-grid">
                        <?php foreach ($testimonials as $testimonial): ?>
                            <div class="testimonial-card">
                                <div class="testimonial-header">
                                    <div class="customer-info">
                                        <h3><?php echo htmlspecialchars($testimonial['name']); ?></h3>
                                        <div class="customer-meta">
                                            <?php if ($testimonial['email']): ?>
                                                <span><i class="fas fa-envelope"></i> <?php echo htmlspecialchars($testimonial['email']); ?></span>
                                            <?php endif; ?>
                                            <?php if ($testimonial['phone']): ?>
                                                <span><i class="fas fa-phone"></i> <?php echo htmlspecialchars($testimonial['phone']); ?></span>
                                            <?php endif; ?>
                                        </div>
                                        <?php if ($testimonial['event_type'] || $testimonial['event_date']): ?>
                                            <div class="customer-meta">
                                                <?php if ($testimonial['event_type']): ?>
                                                    <span><i class="fas fa-calendar"></i> <?php echo htmlspecialchars($testimonial['event_type']); ?></span>
                                                <?php endif; ?>
                                                <?php if ($testimonial['event_date']): ?>
                                                    <span><i class="fas fa-clock"></i> <?php echo date('M d, Y', strtotime($testimonial['event_date'])); ?></span>
                                                <?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                        <div class="rating-display">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <i class="fas fa-star star <?php echo $i <= $testimonial['rating'] ? '' : 'empty'; ?>"></i>
                                            <?php endfor; ?>
                                        </div>
                                    </div>
                                    <div>
                                        <span class="status-badge status-<?php echo $testimonial['status']; ?>">
                                            <?php echo ucfirst($testimonial['status']); ?>
                                        </span>
                                        <?php if ($testimonial['show_on_home']): ?>
                                            <span class="status-badge home-display-badge">On Home</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <div class="testimonial-text">
                                    "<?php echo htmlspecialchars($testimonial['testimonial']); ?>"
                                </div>
                                
                                <?php if (!empty($testimonial['admin_notes'])): ?>
                                    <div class="admin-notes">
                                        <strong>Admin Notes:</strong> <?php echo htmlspecialchars($testimonial['admin_notes']); ?>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="testimonial-footer">
                                    <small class="text-muted">
                                        Submitted: <?php echo date('M d, Y g:i A', strtotime($testimonial['created_at'])); ?>
                                    </small>
                                    
                                    <div class="testimonial-actions">
                                        <a href="testimonials.php?edit=<?php echo $testimonial['id']; ?>" class="btn btn-secondary btn-sm">
                                            <i class="fas fa-edit"></i> Edit
                                        </a>
                                        
                                        <?php if ($testimonial['status'] === 'pending'): ?>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="testimonial_id" value="<?php echo $testimonial['id']; ?>">
                                                <input type="hidden" name="new_status" value="approved">
                                                <button type="submit" name="update_status" class="btn btn-success btn-sm">
                                                    <i class="fas fa-check"></i> Approve
                                                </button>
                                            </form>
                                            
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="testimonial_id" value="<?php echo $testimonial['id']; ?>">
                                                <input type="hidden" name="new_status" value="rejected">
                                                <button type="submit" name="update_status" class="btn btn-danger btn-sm">
                                                    <i class="fas fa-times"></i> Reject
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                        
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="testimonial_id" value="<?php echo $testimonial['id']; ?>">
                                            <input type="hidden" name="value" value="<?php echo $testimonial['show_on_home'] ? 0 : 1; ?>">
                                            <button type="submit" name="toggle_home_display" class="btn btn-warning btn-sm">
                                                <i class="fas fa-home"></i>
                                                <?php echo $testimonial['show_on_home'] ? 'Hide from Home' : 'Show on Home'; ?>
                                            </button>
                                        </form>
                                        
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this testimonial?');">
                                            <input type="hidden" name="testimonial_id" value="<?php echo $testimonial['id']; ?>">
                                            <button type="submit" name="delete_testimonial" class="btn btn-danger btn-sm">
                                                <i class="fas fa-trash"></i> Delete
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>