/**
 * SSV Website - Main JavaScript
 * Shree Shyam Vartika Event Management
 * Interactive features, animations, and functionality
 */

// DOM Ready
document.addEventListener('DOMContentLoaded', function() {
    initializeCommonFeatures();
});

/**
 * Initialize common features across all pages
 */
function initializeCommonFeatures() {
    initializePreloader();
    initializeHeader();
    initializeMobileNavigation();
    initializeScrollToTop();
    initializeAnimations();
    initializeLightbox();
    initializeFormValidation();
    initializeGalleryFilters();
    initializeContactForm();
}

/**
 * Preloader functionality
 */
function initializePreloader() {
    const preloader = document.getElementById('preloader');
    
    if (preloader) {
        // Hide preloader after page load
        window.addEventListener('load', function() {
            setTimeout(() => {
                preloader.classList.add('hidden');
                
                // Remove from DOM after animation
                setTimeout(() => {
                    preloader.style.display = 'none';
                }, 500);
            }, 1000); // Show for at least 1 second
        });
    }
}

/**
 * Header scroll effects and navigation highlighting
 */
function initializeHeader() {
    const header = document.getElementById('header');
    const navLinks = document.querySelectorAll('.nav-link');
    
    if (header) {
        // Header scroll effect
        window.addEventListener('scroll', function() {
            if (window.scrollY > 100) {
                header.classList.add('scrolled');
            } else {
                header.classList.remove('scrolled');
            }
        });
        
        // Smooth scrolling for anchor links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                
                if (target) {
                    const headerHeight = header.offsetHeight;
                    const targetPosition = target.offsetTop - headerHeight - 20;
                    
                    window.scrollTo({
                        top: targetPosition,
                        behavior: 'smooth'
                    });
                }
            });
        });
    }
    
    // Active navigation highlighting based on scroll position
    window.addEventListener('scroll', highlightActiveSection);
}

/**
 * Highlight active navigation section
 */
function highlightActiveSection() {
    const sections = document.querySelectorAll('section[id]');
    const navLinks = document.querySelectorAll('.nav-link[href^="#"]');
    let current = '';
    
    sections.forEach(section => {
        const sectionTop = section.offsetTop - 150;
        const sectionHeight = section.offsetHeight;
        
        if (window.scrollY >= sectionTop && window.scrollY < sectionTop + sectionHeight) {
            current = section.getAttribute('id');
        }
    });
    
    navLinks.forEach(link => {
        link.classList.remove('active');
        if (link.getAttribute('href') === '#' + current) {
            link.classList.add('active');
        }
    });
}

/**
 * Mobile navigation functionality
 */
function initializeMobileNavigation() {
    const mobileToggle = document.getElementById('mobileMenuToggle');
    const mobileOverlay = document.getElementById('mobileNavOverlay');
    const mobileClose = document.getElementById('mobileNavClose');
    const mobileLinks = document.querySelectorAll('.mobile-nav-menu a');
    
    if (mobileToggle && mobileOverlay) {
        // Open mobile menu
        mobileToggle.addEventListener('click', function() {
            mobileToggle.classList.add('active');
            mobileOverlay.classList.add('active');
            document.body.classList.add('mobile-nav-open');
        });
        
        // Close mobile menu
        function closeMobileMenu() {
            mobileToggle.classList.remove('active');
            mobileOverlay.classList.remove('active');
            document.body.classList.remove('mobile-nav-open');
        }
        
        if (mobileClose) {
            mobileClose.addEventListener('click', closeMobileMenu);
        }
        
        // Close on overlay click
        mobileOverlay.addEventListener('click', function(e) {
            if (e.target === mobileOverlay) {
                closeMobileMenu();
            }
        });
        
        // Close on link click
        mobileLinks.forEach(link => {
            link.addEventListener('click', closeMobileMenu);
        });
        
        // Close on escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && mobileOverlay.classList.contains('active')) {
                closeMobileMenu();
            }
        });
    }
}

/**
 * Scroll to top button functionality
 */
function initializeScrollToTop() {
    const scrollButton = document.getElementById('scrollToTop');
    
    if (scrollButton) {
        // Show/hide scroll button
        window.addEventListener('scroll', function() {
            if (window.scrollY > 300) {
                scrollButton.classList.add('visible');
            } else {
                scrollButton.classList.remove('visible');
            }
        });
        
        // Scroll to top on click
        scrollButton.addEventListener('click', function() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        });
    }
}

/**
 * Animation on scroll functionality
 */
function initializeAnimations() {
    const animatedElements = document.querySelectorAll('.animate-fade-up, .animate-fade-left, .animate-fade-right');
    
    // Intersection Observer for scroll animations
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0) translateX(0)';
            }
        });
    }, {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    });
    
    animatedElements.forEach(element => {
        // Set initial state
        element.style.opacity = '0';
        if (element.classList.contains('animate-fade-up')) {
            element.style.transform = 'translateY(30px)';
        } else if (element.classList.contains('animate-fade-left')) {
            element.style.transform = 'translateX(-30px)';
        } else if (element.classList.contains('animate-fade-right')) {
            element.style.transform = 'translateX(30px)';
        }
        
        element.style.transition = 'opacity 0.6s ease-out, transform 0.6s ease-out';
        observer.observe(element);
    });
    
    // Staggered animation for card grids
    const cardGrids = document.querySelectorAll('.grid');
    cardGrids.forEach(grid => {
        const cards = grid.querySelectorAll('.card, .service-card, .gallery-item');
        cards.forEach((card, index) => {
            card.style.animationDelay = `${index * 0.1}s`;
        });
    });
}

/**
 * Lightbox functionality for gallery
 */
function initializeLightbox() {
    const galleryItems = document.querySelectorAll('.gallery-item');
    
    // Create lightbox HTML
    const lightboxHTML = `
        <div id="lightbox" class="lightbox">
            <div class="lightbox-overlay"></div>
            <div class="lightbox-content">
                <button class="lightbox-close">&times;</button>
                <button class="lightbox-prev">&#8249;</button>
                <button class="lightbox-next">&#8250;</button>
                <div class="lightbox-media"></div>
                <div class="lightbox-caption"></div>
            </div>
        </div>
    `;
    
    // Add lightbox to body if gallery items exist
    if (galleryItems.length > 0) {
        document.body.insertAdjacentHTML('beforeend', lightboxHTML);
        
        const lightbox = document.getElementById('lightbox');
        const lightboxMedia = lightbox.querySelector('.lightbox-media');
        const lightboxCaption = lightbox.querySelector('.lightbox-caption');
        const closeBtn = lightbox.querySelector('.lightbox-close');
        const prevBtn = lightbox.querySelector('.lightbox-prev');
        const nextBtn = lightbox.querySelector('.lightbox-next');
        
        let currentIndex = 0;
        let mediaItems = [];
        
        // Collect all media items
        galleryItems.forEach((item, index) => {
            const img = item.querySelector('img');
            const video = item.querySelector('video');
            
            if (img || video) {
                const mediaData = {
                    type: img ? 'image' : 'video',
                    src: img ? img.src : video.src,
                    alt: img ? img.alt : video.title || '',
                    caption: item.dataset.caption || ''
                };
                
                mediaItems.push(mediaData);
                
                // Add click event to gallery item
                item.addEventListener('click', function() {
                    currentIndex = index;
                    openLightbox();
                });
            }
        });
        
        function openLightbox() {
            lightbox.classList.add('active');
            document.body.style.overflow = 'hidden';
            showMedia(currentIndex);
        }
        
        function closeLightbox() {
            lightbox.classList.remove('active');
            document.body.style.overflow = '';
        }
        
        function showMedia(index) {
            if (index < 0 || index >= mediaItems.length) return;
            
            const media = mediaItems[index];
            lightboxMedia.innerHTML = '';
            
            if (media.type === 'image') {
                const img = document.createElement('img');
                img.src = media.src;
                img.alt = media.alt;
                lightboxMedia.appendChild(img);
            } else if (media.type === 'video') {
                const video = document.createElement('video');
                video.src = media.src;
                video.controls = true;
                video.autoplay = true;
                lightboxMedia.appendChild(video);
            }
            
            lightboxCaption.textContent = media.caption || media.alt;
            
            // Update navigation buttons
            prevBtn.style.display = index > 0 ? 'block' : 'none';
            nextBtn.style.display = index < mediaItems.length - 1 ? 'block' : 'none';
        }
        
        // Event listeners
        closeBtn.addEventListener('click', closeLightbox);
        lightbox.querySelector('.lightbox-overlay').addEventListener('click', closeLightbox);
        
        prevBtn.addEventListener('click', function() {
            if (currentIndex > 0) {
                currentIndex--;
                showMedia(currentIndex);
            }
        });
        
        nextBtn.addEventListener('click', function() {
            if (currentIndex < mediaItems.length - 1) {
                currentIndex++;
                showMedia(currentIndex);
            }
        });
        
        // Keyboard navigation
        document.addEventListener('keydown', function(e) {
            if (!lightbox.classList.contains('active')) return;
            
            switch(e.key) {
                case 'Escape':
                    closeLightbox();
                    break;
                case 'ArrowLeft':
                    if (currentIndex > 0) {
                        currentIndex--;
                        showMedia(currentIndex);
                    }
                    break;
                case 'ArrowRight':
                    if (currentIndex < mediaItems.length - 1) {
                        currentIndex++;
                        showMedia(currentIndex);
                    }
                    break;
            }
        });
    }
    
    // Add lightbox styles
    const lightboxStyles = `
        <style>
        .lightbox {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .lightbox.active {
            opacity: 1;
            visibility: visible;
        }
        
        .lightbox-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(10px);
        }
        
        .lightbox-content {
            position: relative;
            max-width: 90vw;
            max-height: 90vh;
            z-index: 2;
        }
        
        .lightbox-media {
            display: flex;
            align-items: center;
            justify-content: center;
            max-width: 100%;
            max-height: 80vh;
        }
        
        .lightbox-media img,
        .lightbox-media video {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: 8px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.5);
        }
        
        .lightbox-caption {
            text-align: center;
            color: white;
            margin-top: 20px;
            font-size: 1.1rem;
        }
        
        .lightbox-close,
        .lightbox-prev,
        .lightbox-next {
            position: absolute;
            background: rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
            border: none;
            color: white;
            font-size: 2rem;
            cursor: pointer;
            padding: 10px 15px;
            border-radius: 50%;
            transition: all 0.3s ease;
        }
        
        .lightbox-close:hover,
        .lightbox-prev:hover,
        .lightbox-next:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: scale(1.1);
        }
        
        .lightbox-close {
            top: -50px;
            right: 0;
            font-size: 2.5rem;
            padding: 5px 15px;
        }
        
        .lightbox-prev {
            top: 50%;
            left: -60px;
            transform: translateY(-50%);
            font-size: 3rem;
            padding: 15px 20px;
        }
        
        .lightbox-next {
            top: 50%;
            right: -60px;
            transform: translateY(-50%);
            font-size: 3rem;
            padding: 15px 20px;
        }
        
        @media (max-width: 768px) {
            .lightbox-prev {
                left: 10px;
                font-size: 2rem;
                padding: 10px 15px;
            }
            
            .lightbox-next {
                right: 10px;
                font-size: 2rem;
                padding: 10px 15px;
            }
            
            .lightbox-close {
                top: 10px;
                right: 10px;
                font-size: 2rem;
            }
        }
        </style>
    `;
    
    document.head.insertAdjacentHTML('beforeend', lightboxStyles);
}

/**
 * Form validation functionality
 */
function initializeFormValidation() {
    const forms = document.querySelectorAll('form[data-validate="true"]');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(form);
            const errors = validateForm(form);
            
            // Clear previous error messages
            clearErrorMessages(form);
            
            if (errors.length === 0) {
                // Form is valid, submit it
                submitForm(form, formData);
            } else {
                // Show error messages
                showErrorMessages(form, errors);
            }
        });
        
        // Real-time validation
        const inputs = form.querySelectorAll('input, textarea, select');
        inputs.forEach(input => {
            input.addEventListener('blur', function() {
                validateField(input);
            });
            
            input.addEventListener('input', function() {
                clearFieldError(input);
            });
        });
    });
}

/**
 * Validate form fields
 */
function validateForm(form) {
    const errors = [];
    const requiredFields = form.querySelectorAll('[required]');
    
    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            errors.push({
                field: field,
                message: `${getFieldLabel(field)} is required`
            });
        } else {
            // Validate field based on type
            const fieldErrors = validateField(field);
            errors.push(...fieldErrors);
        }
    });
    
    return errors;
}

/**
 * Validate individual field
 */
function validateField(field) {
    const errors = [];
    const value = field.value.trim();
    const fieldName = getFieldLabel(field);
    
    if (!value && field.hasAttribute('required')) {
        return [{
            field: field,
            message: `${fieldName} is required`
        }];
    }
    
    if (value) {
        switch (field.type) {
            case 'email':
                if (!isValidEmail(value)) {
                    errors.push({
                        field: field,
                        message: 'Please enter a valid email address'
                    });
                }
                break;
            case 'tel':
                if (!isValidPhone(value)) {
                    errors.push({
                        field: field,
                        message: 'Please enter a valid phone number'
                    });
                }
                break;
            case 'url':
                if (!isValidURL(value)) {
                    errors.push({
                        field: field,
                        message: 'Please enter a valid URL'
                    });
                }
                break;
        }
        
        // Check minimum length
        if (field.hasAttribute('minlength')) {
            const minLength = parseInt(field.getAttribute('minlength'));
            if (value.length < minLength) {
                errors.push({
                    field: field,
                    message: `${fieldName} must be at least ${minLength} characters long`
                });
            }
        }
        
        // Check maximum length
        if (field.hasAttribute('maxlength')) {
            const maxLength = parseInt(field.getAttribute('maxlength'));
            if (value.length > maxLength) {
                errors.push({
                    field: field,
                    message: `${fieldName} must not exceed ${maxLength} characters`
                });
            }
        }
    }
    
    return errors;
}

/**
 * Validation helper functions
 */
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

function isValidPhone(phone) {
    const phoneRegex = /^[\+]?[1-9][\d]{0,15}$/;
    return phoneRegex.test(phone.replace(/[\s\-\(\)]/g, ''));
}

function isValidURL(url) {
    try {
        new URL(url);
        return true;
    } catch {
        return false;
    }
}

function getFieldLabel(field) {
    const label = field.closest('.form-group')?.querySelector('label');
    return label ? label.textContent.replace('*', '').trim() : field.name || 'Field';
}

/**
 * Error message handling
 */
function showErrorMessages(form, errors) {
    errors.forEach(error => {
        const field = error.field;
        const formGroup = field.closest('.form-group');
        
        if (formGroup) {
            const errorElement = document.createElement('div');
            errorElement.className = 'error-message';
            errorElement.textContent = error.message;
            formGroup.appendChild(errorElement);
            
            field.classList.add('error');
        }
    });
}

function clearErrorMessages(form) {
    const errorMessages = form.querySelectorAll('.error-message');
    const errorFields = form.querySelectorAll('.error');
    
    errorMessages.forEach(msg => msg.remove());
    errorFields.forEach(field => field.classList.remove('error'));
}

function clearFieldError(field) {
    const formGroup = field.closest('.form-group');
    if (formGroup) {
        const errorMessage = formGroup.querySelector('.error-message');
        if (errorMessage) {
            errorMessage.remove();
        }
        field.classList.remove('error');
    }
}

/**
 * Form submission
 */
function submitForm(form, formData) {
    const submitButton = form.querySelector('button[type="submit"]');
    const originalText = submitButton.textContent;
    
    // Show loading state
    submitButton.disabled = true;
    submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';
    
    // Submit form via AJAX
    fetch(form.action || window.location.href, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Message sent successfully!', 'success');
            form.reset();
        } else {
            showNotification(data.message || 'An error occurred. Please try again.', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('An error occurred. Please try again.', 'error');
    })
    .finally(() => {
        // Reset button state
        submitButton.disabled = false;
        submitButton.textContent = originalText;
    });
}

/**
 * Gallery filtering functionality
 */
function initializeGalleryFilters() {
    const filterButtons = document.querySelectorAll('.filter-btn');
    const galleryItems = document.querySelectorAll('.gallery-item');
    
    if (filterButtons.length > 0 && galleryItems.length > 0) {
        filterButtons.forEach(button => {
            button.addEventListener('click', function() {
                const filter = this.getAttribute('data-filter');
                
                // Update active button
                filterButtons.forEach(btn => btn.classList.remove('active'));
                this.classList.add('active');
                
                // Scroll the active button into view on mobile/tablet
                if (window.innerWidth <= 768) {
                    setTimeout(() => {
                        this.scrollIntoView({ 
                            behavior: 'smooth', 
                            block: 'nearest', 
                            inline: 'center' 
                        });
                    }, 100);
                }
                
                // Filter gallery items
                galleryItems.forEach(item => {
                    const category = item.getAttribute('data-category');
                    
                    if (filter === 'all' || category === filter) {
                        item.style.display = 'block';
                        setTimeout(() => {
                            item.style.opacity = '1';
                            item.style.transform = 'scale(1)';
                        }, 50);
                    } else {
                        item.style.opacity = '0';
                        item.style.transform = 'scale(0.8)';
                        setTimeout(() => {
                            item.style.display = 'none';
                        }, 300);
                    }
                });
            });
        });
    }
    
    // Initialize smooth scrolling for filter containers
    initializeFilterScroll();
}

/**
 * Enhanced scrolling for filter buttons on mobile/tablet
 */
function initializeFilterScroll() {
    const filterContainers = document.querySelectorAll('.category-filters, .gallery-filters');
    
    filterContainers.forEach(container => {
        if (window.innerWidth <= 768) {
            // Enable smooth scrolling
            container.style.scrollBehavior = 'smooth';
            
            // Add scroll momentum on touch devices
            container.style.webkitOverflowScrolling = 'touch';
            
            // Handle button clicks to center the active button
            const buttons = container.querySelectorAll('.filter-btn');
            buttons.forEach(button => {
                button.addEventListener('click', function() {
                    setTimeout(() => {
                        // Center the clicked button in the scroll area
                        const containerRect = container.getBoundingClientRect();
                        const buttonRect = this.getBoundingClientRect();
                        const scrollLeft = this.offsetLeft - (containerRect.width / 2) + (buttonRect.width / 2);
                        
                        container.scrollTo({
                            left: scrollLeft,
                            behavior: 'smooth'
                        });
                    }, 150);
                });
            });
        }
    });
}

/**
 * Contact form specific functionality
 */
function initializeContactForm() {
    const contactForm = document.getElementById('contactForm');
    
    if (contactForm) {
        // Date picker for event date
        const eventDateInput = contactForm.querySelector('input[type="date"]');
        if (eventDateInput) {
            // Set minimum date to today
            const today = new Date().toISOString().split('T')[0];
            eventDateInput.setAttribute('min', today);
        }
        
        // Auto-format phone number
        const phoneInput = contactForm.querySelector('input[type="tel"]');
        if (phoneInput) {
            phoneInput.addEventListener('input', function() {
                let value = this.value.replace(/\D/g, '');
                if (value.length >= 10) {
                    value = value.replace(/(\d{3})(\d{3})(\d{4})/, '($1) $2-$3');
                }
                this.value = value;
            });
        }
    }
}

/**
 * Notification system
 */
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas fa-${getNotificationIcon(type)}"></i>
            <span>${message}</span>
            <button class="notification-close">&times;</button>
        </div>
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Show notification
    setTimeout(() => {
        notification.classList.add('show');
    }, 100);
    
    // Auto hide after 5 seconds
    setTimeout(() => {
        hideNotification(notification);
    }, 5000);
    
    // Close button
    notification.querySelector('.notification-close').addEventListener('click', () => {
        hideNotification(notification);
    });
}

function hideNotification(notification) {
    notification.classList.remove('show');
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 300);
}

function getNotificationIcon(type) {
    switch (type) {
        case 'success': return 'check-circle';
        case 'error': return 'exclamation-circle';
        case 'warning': return 'exclamation-triangle';
        default: return 'info-circle';
    }
}

// Add notification styles
const notificationStyles = `
    <style>
    .notification {
        position: fixed;
        top: 20px;
        right: 20px;
        background: white;
        border-radius: 8px;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        backdrop-filter: blur(10px);
        z-index: 10000;
        max-width: 400px;
        transform: translateX(400px);
        transition: all 0.3s ease;
    }
    
    .notification.show {
        transform: translateX(0);
    }
    
    .notification-content {
        display: flex;
        align-items: center;
        gap: 12px;
        padding: 16px 20px;
    }
    
    .notification-success {
        border-left: 4px solid #10B981;
    }
    
    .notification-error {
        border-left: 4px solid #EF4444;
    }
    
    .notification-warning {
        border-left: 4px solid #F59E0B;
    }
    
    .notification-info {
        border-left: 4px solid #3B82F6;
    }
    
    .notification i {
        font-size: 1.2rem;
    }
    
    .notification-success i { color: #10B981; }
    .notification-error i { color: #EF4444; }
    .notification-warning i { color: #F59E0B; }
    .notification-info i { color: #3B82F6; }
    
    .notification-close {
        background: none;
        border: none;
        font-size: 1.5rem;
        cursor: pointer;
        color: #6B7280;
        margin-left: auto;
    }
    
    .notification-close:hover {
        color: #374151;
    }
    
    @media (max-width: 768px) {
        .notification {
            right: 10px;
            left: 10px;
            max-width: none;
            transform: translateY(-100px);
        }
        
        .notification.show {
            transform: translateY(0);
        }
    }
    </style>
`;

document.head.insertAdjacentHTML('beforeend', notificationStyles);

// Error message styles
const errorStyles = `
    <style>
    .form-control.error {
        border-color: #EF4444;
        background-color: #FEF2F2;
    }
    
    .error-message {
        color: #EF4444;
        font-size: 0.875rem;
        margin-top: 4px;
        display: flex;
        align-items: center;
        gap: 4px;
    }
    
    .error-message::before {
        content: '⚠️';
        font-size: 0.8rem;
    }
    </style>
`;

document.head.insertAdjacentHTML('beforeend', errorStyles);

/**
 * Utility functions
 */
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

function throttle(func, limit) {
    let inThrottle;
    return function() {
        const args = arguments;
        const context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    }
}

// Performance optimization
window.addEventListener('scroll', throttle(function() {
    // Throttled scroll events are handled in individual functions
}, 16)); // ~60fps

// Image lazy loading for better performance
function initializeLazyLoading() {
    const lazyImages = document.querySelectorAll('img[data-src]');
    
    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    img.src = img.dataset.src;
                    img.classList.remove('lazy');
                    imageObserver.unobserve(img);
                }
            });
        });
        
        lazyImages.forEach(img => imageObserver.observe(img));
    } else {
        // Fallback for older browsers
        lazyImages.forEach(img => {
            img.src = img.dataset.src;
            img.classList.remove('lazy');
        });
    }
}

// Initialize lazy loading when DOM is ready
document.addEventListener('DOMContentLoaded', initializeLazyLoading);

// Export functions for external use
window.SSV = {
    showNotification,
    initializeCommonFeatures,
    validateForm,
    submitForm
};