<?php
// Page configuration
$page_title = "Contact Us";
$page_description = "Get in touch with Shree Shyam Vartika for your event management needs. Contact us for free consultation and quotes for weddings, engagements, birthdays, and corporate events.";
$body_class = "contact-page";

// Include database connection
require_once '../config/db.php';

// Handle form submission
$form_submitted = false;
$form_success = false;
$form_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $form_submitted = true;
    
    // Sanitize input data
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $service = trim($_POST['service'] ?? '');
    $event_date = trim($_POST['event_date'] ?? '');
    $guest_count = trim($_POST['guest_count'] ?? '');
    $message = trim($_POST['message'] ?? '');
    
    // Basic validation
    $errors = [];
    
    if (empty($name)) $errors[] = 'Name is required';
    if (empty($email)) $errors[] = 'Email is required';
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Valid email is required';
    if (empty($phone)) $errors[] = 'Phone number is required';
    if (empty($message)) $errors[] = 'Message is required';
    
    if (empty($errors)) {
        try {
            $db = getDB();
            $stmt = $db->prepare("INSERT INTO enquiries (name, email, phone, service_interested, event_date, guest_count, message) VALUES (?, ?, ?, ?, ?, ?, ?)");
            
            $event_date_formatted = !empty($event_date) ? $event_date : null;
            $guest_count_int = !empty($guest_count) ? intval($guest_count) : null;
            
            $stmt->execute([$name, $email, $phone, $service, $event_date_formatted, $guest_count_int, $message]);
            
            $form_success = true;
            $form_message = 'Thank you for your inquiry! We will get back to you soon.';
            
            // If AJAX request, return JSON
            if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                header('Content-Type: application/json');
                echo json_encode(['success' => true, 'message' => $form_message]);
                exit;
            }
            
        } catch (Exception $e) {
            $form_success = false;
            $form_message = 'Sorry, there was an error processing your request. Please try again.';
            
            if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'message' => $form_message]);
                exit;
            }
        }
    } else {
        $form_message = implode(', ', $errors);
        
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'message' => $form_message]);
            exit;
        }
    }
}

// Get business settings for contact info
$business_phone = '+91-9876543210';
$business_email = 'info@shreeshyamvartika.com';
$business_address = '123 Event Street, Wedding City, State 123456';
$whatsapp_number = '+91-9876543210';
$google_maps_embed = 'https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3024.123456789!2d-74.00597238459394!3d40.712775279337!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x0%3A0x0!2zNDDCsDQyJzQ2LjAiTiA3NMKwMDAnMjEuNSJX!5e0!3m2!1sen!2sin!4v1234567890123!5m2!1sen!2sin';

try {
    $db = getDB();
    $stmt = $db->prepare("SELECT setting_key, setting_value FROM business_settings WHERE setting_key IN ('business_phone', 'business_email', 'business_address', 'whatsapp_number', 'google_maps_embed')");
    $stmt->execute();
    $settings = $stmt->fetchAll();
    
    foreach ($settings as $setting) {
        switch ($setting['setting_key']) {
            case 'business_phone':
                $business_phone = $setting['setting_value'];
                break;
            case 'business_email':
                $business_email = $setting['setting_value'];
                break;
            case 'business_address':
                $business_address = $setting['setting_value'];
                break;
            case 'whatsapp_number':
                $whatsapp_number = $setting['setting_value'];
                break;
            case 'google_maps_embed':
                $google_maps_embed = $setting['setting_value'];
                break;
        }
    }
} catch (Exception $e) {
    // Use default values
}

// Include header
include '../includes/header.php';
?>

<!-- Contact Hero Section -->
<section class="section" style="background: var(--bg-gradient); color: white; padding-top: 8rem;">
    <div class="container">
        <div class="text-center animate-fade-up">
            <h1 class="section-title" style="color: white; font-size: 3rem;">Contact Us</h1>
            <p class="section-subtitle script-text" style="color: var(--light-gold); font-size: 1.5rem;">Let's Plan Your Perfect Event Together</p>
            <p class="section-description" style="color: rgba(255,255,255,0.9); font-size: 1.2rem; margin-bottom: 3rem;">
                Ready to create unforgettable memories? Get in touch with us for a free consultation and personalized quote for your special event.
            </p>
        </div>
    </div>
</section>

<!-- Contact Form & Info Section -->
<section class="section">
    <div class="container">
        <div class="grid grid-2" style="gap: 4rem;">
            <!-- Contact Form -->
            <div class="animate-fade-left">
                <div class="card glass-card" style="padding: 3rem;">
                    <h3 style="margin-bottom: 1.5rem; color: var(--primary-purple);">Send Us a Message</h3>
                    
                    <?php if ($form_submitted && !empty($form_message)): ?>
                    <div class="form-message <?php echo $form_success ? 'success' : 'error'; ?>" style="margin-bottom: 1.5rem; padding: 1rem; border-radius: 8px; <?php echo $form_success ? 'background: #D1FAE5; color: #065F46; border: 1px solid #10B981;' : 'background: #FEE2E2; color: #991B1B; border: 1px solid #EF4444;'; ?>">
                        <i class="fas fa-<?php echo $form_success ? 'check-circle' : 'exclamation-circle'; ?>"></i>
                        <?php echo htmlspecialchars($form_message); ?>
                    </div>
                    <?php endif; ?>
                    
                    <form id="contactForm" method="POST" data-validate="true">
                        <div class="grid grid-2" style="gap: 1rem;">
                            <div class="form-group">
                                <label for="name" class="form-label">Full Name *</label>
                                <input type="text" id="name" name="name" class="form-control" required 
                                       value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>"
                                       placeholder="Enter your full name">
                            </div>
                            
                            <div class="form-group">
                                <label for="email" class="form-label">Email Address *</label>
                                <input type="email" id="email" name="email" class="form-control" required 
                                       value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>"
                                       placeholder="Enter your email address">
                            </div>
                        </div>
                        
                        <div class="grid grid-2" style="gap: 1rem;">
                            <div class="form-group">
                                <label for="phone" class="form-label">Phone Number *</label>
                                <input type="tel" id="phone" name="phone" class="form-control" required 
                                       value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>"
                                       placeholder="Enter your phone number">
                            </div>
                            
                            <div class="form-group">
                                <label for="service" class="form-label">Service Interested</label>
                                <select id="service" name="service" class="form-control">
                                    <option value="">Select a service</option>
                                    <option value="Wedding" <?php echo ($_POST['service'] ?? '') === 'Wedding' ? 'selected' : ''; ?>>Wedding Planning</option>
                                    <option value="Engagement" <?php echo ($_POST['service'] ?? '') === 'Engagement' ? 'selected' : ''; ?>>Engagement Ceremony</option>
                                    <option value="Reception" <?php echo ($_POST['service'] ?? '') === 'Reception' ? 'selected' : ''; ?>>Reception Party</option>
                                    <option value="Birthday" <?php echo ($_POST['service'] ?? '') === 'Birthday' ? 'selected' : ''; ?>>Birthday Celebration</option>
                                    <option value="Corporate" <?php echo ($_POST['service'] ?? '') === 'Corporate' ? 'selected' : ''; ?>>Corporate Event</option>
                                    <option value="Anniversary" <?php echo ($_POST['service'] ?? '') === 'Anniversary' ? 'selected' : ''; ?>>Anniversary</option>
                                    <option value="Other" <?php echo ($_POST['service'] ?? '') === 'Other' ? 'selected' : ''; ?>>Other</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="grid grid-2" style="gap: 1rem;">
                            <div class="form-group">
                                <label for="event_date" class="form-label">Event Date</label>
                                <input type="date" id="event_date" name="event_date" class="form-control" 
                                       value="<?php echo htmlspecialchars($_POST['event_date'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="guest_count" class="form-label">Expected Guest Count</label>
                                <input type="number" id="guest_count" name="guest_count" class="form-control" min="1" max="2000"
                                       value="<?php echo htmlspecialchars($_POST['guest_count'] ?? ''); ?>"
                                       placeholder="Number of guests">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="message" class="form-label">Message *</label>
                            <textarea id="message" name="message" class="form-control" rows="5" required 
                                      placeholder="Tell us about your event requirements, preferences, and any specific details..."><?php echo htmlspecialchars($_POST['message'] ?? ''); ?></textarea>
                        </div>
                        
                        <button type="submit" class="btn btn-primary btn-lg" style="width: 100%;">
                            <i class="fas fa-paper-plane"></i>
                            Send Message
                        </button>
                    </form>
                </div>
            </div>
            
            <!-- Contact Information -->
            <div class="animate-fade-right">
                <div class="contact-info-section" style="margin-bottom: 3rem;">
                    <h3 style="margin-bottom: 2rem; color: var(--primary-purple);">Get In Touch</h3>
                    
                    <div class="contact-item" style="display: flex; align-items: flex-start; gap: 1rem; margin-bottom: 2rem;">
                        <div class="contact-icon" style="width: 50px; height: 50px; background: var(--glass-white); border-radius: 50%; display: flex; align-items: center; justify-content: center; color: var(--primary-purple); font-size: 1.2rem;">
                            <i class="fas fa-map-marker-alt"></i>
                        </div>
                        <div>
                            <h5 style="margin-bottom: 0.5rem; color: var(--text-primary);">Our Office</h5>
                            <p style="color: var(--text-secondary); margin: 0;"><?php echo htmlspecialchars($business_address); ?></p>
                        </div>
                    </div>
                    
                    <div class="contact-item" style="display: flex; align-items: flex-start; gap: 1rem; margin-bottom: 2rem;">
                        <div class="contact-icon" style="width: 50px; height: 50px; background: var(--glass-white); border-radius: 50%; display: flex; align-items: center; justify-content: center; color: var(--primary-purple); font-size: 1.2rem;">
                            <i class="fas fa-phone"></i>
                        </div>
                        <div>
                            <h5 style="margin-bottom: 0.5rem; color: var(--text-primary);">Phone Number</h5>
                            <p style="color: var(--text-secondary); margin: 0;">
                                <a href="tel:<?php echo str_replace(['+', '-', ' '], '', $business_phone); ?>" 
                                   style="color: var(--primary-purple); text-decoration: none;">
                                   <?php echo htmlspecialchars($business_phone); ?>
                                </a>
                            </p>
                        </div>
                    </div>
                    
                    <div class="contact-item" style="display: flex; align-items: flex-start; gap: 1rem; margin-bottom: 2rem;">
                        <div class="contact-icon" style="width: 50px; height: 50px; background: var(--glass-white); border-radius: 50%; display: flex; align-items: center; justify-content: center; color: var(--primary-purple); font-size: 1.2rem;">
                            <i class="fas fa-envelope"></i>
                        </div>
                        <div>
                            <h5 style="margin-bottom: 0.5rem; color: var(--text-primary);">Email Address</h5>
                            <p style="color: var(--text-secondary); margin: 0;">
                                <a href="mailto:<?php echo htmlspecialchars($business_email); ?>" 
                                   style="color: var(--primary-purple); text-decoration: none;">
                                   <?php echo htmlspecialchars($business_email); ?>
                                </a>
                            </p>
                        </div>
                    </div>
                    
                    <div class="contact-item" style="display: flex; align-items: flex-start; gap: 1rem; margin-bottom: 2rem;">
                        <div class="contact-icon" style="width: 50px; height: 50px; background: #25D366; border-radius: 50%; display: flex; align-items: center; justify-content: center; color: white; font-size: 1.2rem;">
                            <i class="fab fa-whatsapp"></i>
                        </div>
                        <div>
                            <h5 style="margin-bottom: 0.5rem; color: var(--text-primary);">WhatsApp</h5>
                            <p style="color: var(--text-secondary); margin: 0;">
                                <a href="https://wa.me/<?php echo str_replace(['+', '-', ' '], '', $whatsapp_number); ?>?text=Hi%20Shree%20Shyam%20Vartika,%20I%20would%20like%20to%20inquire%20about%20your%20event%20services." 
                                   target="_blank" style="color: #25D366; text-decoration: none;">
                                   Chat with us instantly
                                </a>
                            </p>
                        </div>
                    </div>
                </div>
                
                <!-- Quick Contact Buttons -->
                <div class="quick-contact-buttons" style="display: flex; flex-direction: column; gap: 1rem;">
                    <a href="tel:<?php echo str_replace(['+', '-', ' '], '', $business_phone); ?>" 
                       class="btn btn-primary btn-lg">
                        <i class="fas fa-phone"></i>
                        Call Now
                    </a>
                    <a href="https://wa.me/<?php echo str_replace(['+', '-', ' '], '', $whatsapp_number); ?>?text=Hi%20Shree%20Shyam%20Vartika,%20I%20would%20like%20to%20inquire%20about%20your%20event%20services." 
                       target="_blank" class="btn btn-secondary btn-lg">
                        <i class="fab fa-whatsapp"></i>
                        WhatsApp Us
                    </a>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Google Maps Section -->
<section class="section" style="background: var(--bg-secondary);">
    <div class="container">
        <div class="text-center animate-fade-up" style="margin-bottom: 3rem;">
            <h3 class="section-title">Find Us</h3>
            <p class="section-description">Visit our office for personal consultation and event planning discussions.</p>
        </div>
        
        <div class="map-container animate-fade-up" style="position: relative; width: 100%; height: 400px; border-radius: 16px; overflow: hidden; box-shadow: var(--shadow-lg);">
            <iframe src="<?php echo htmlspecialchars($google_maps_embed); ?>" 
                    width="100%" 
                    height="100%" 
                    style="border:0;" 
                    allowfullscreen="" 
                    loading="lazy" 
                    referrerpolicy="no-referrer-when-downgrade">
            </iframe>
        </div>
        
        <div class="text-center" style="margin-top: 2rem;">
            <a href="https://maps.google.com/maps?q=<?php echo urlencode($business_address); ?>" 
               target="_blank" class="btn btn-outline">
                <i class="fas fa-directions"></i>
                Get Directions
            </a>
        </div>
    </div>
</section>

<!-- Business Hours & Additional Info -->
<section class="section">
    <div class="container">
        <div class="grid grid-3">
            <div class="card glass-card animate-fade-up" style="text-align: center; padding: 2rem;">
                <div style="font-size: 3rem; color: var(--primary-gold); margin-bottom: 1rem;">
                    <i class="fas fa-clock"></i>
                </div>
                <h4 style="margin-bottom: 1rem; color: var(--text-primary);">Business Hours</h4>
                <p style="color: var(--text-secondary); margin: 0; line-height: 1.6;">
                    Monday - Saturday: 9:00 AM - 8:00 PM<br>
                    Sunday: 10:00 AM - 6:00 PM<br>
                    <em>Available 24/7 for urgent inquiries</em>
                </p>
            </div>
            
            <div class="card glass-card animate-fade-up" style="text-align: center; padding: 2rem;">
                <div style="font-size: 3rem; color: var(--primary-gold); margin-bottom: 1rem;">
                    <i class="fas fa-handshake"></i>
                </div>
                <h4 style="margin-bottom: 1rem; color: var(--text-primary);">Free Consultation</h4>
                <p style="color: var(--text-secondary); margin: 0; line-height: 1.6;">
                    Schedule a free consultation to discuss your event vision and get personalized recommendations for your special day.
                </p>
            </div>
            
            <div class="card glass-card animate-fade-up" style="text-align: center; padding: 2rem;">
                <div style="font-size: 3rem; color: var(--primary-gold); margin-bottom: 1rem;">
                    <i class="fas fa-award"></i>
                </div>
                <h4 style="margin-bottom: 1rem; color: var(--text-primary);">Satisfaction Guaranteed</h4>
                <p style="color: var(--text-secondary); margin: 0; line-height: 1.6;">
                    We guarantee 100% satisfaction with our services. Your perfect event is our commitment to you.
                </p>
            </div>
        </div>
    </div>
</section>

<style>
/* Additional styles for contact page */
.text-center { text-align: center; }

.form-message {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

@media (max-width: 768px) {
    .quick-contact-buttons {
        margin-top: 2rem;
    }
    
    .contact-item {
        flex-direction: column;
        text-align: center;
        margin-bottom: 1.5rem !important;
    }
    
    .map-container {
        height: 300px !important;
    }
}
</style>

<?php include '../includes/footer.php'; ?>