<?php
// Page configuration
$page_title = "Gallery";
$page_description = "View our stunning collection of event photos and videos. Browse through weddings, engagements, birthdays, corporate events, and more memorable celebrations.";
$body_class = "gallery-page";

// Include database connection
require_once '../config/db.php';

// Fetch gallery items
$gallery_items = [];
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM gallery ORDER BY display_order, created_at DESC");
    $stmt->execute();
    $gallery_items = $stmt->fetchAll();
} catch(Exception $e) {
    // Handle error silently for now
}

// Fetch service categories for filtering
$categories = [];
try {
    $stmt = $db->prepare("SELECT name FROM service_categories WHERE is_active = 1 ORDER BY name");
    $stmt->execute();
    $categories = $stmt->fetchAll(PDO::FETCH_COLUMN);
} catch(Exception $e) {
    // Handle error silently for now
}

// Include header
include '../includes/header.php';
?>

<!-- Gallery Hero Section -->
<section class="section" style="background: var(--bg-gradient); color: white; padding-top: 8rem;">
    <div class="container">
        <div class="text-center animate-fade-up">
            <h1 class="section-title" style="color: white; font-size: 3rem;">Event Gallery</h1>
            <p class="section-subtitle script-text" style="color: var(--light-gold); font-size: 1.5rem;">Memories That Last Forever</p>
            <p class="section-description" style="color: rgba(255,255,255,0.9); font-size: 1.2rem; margin-bottom: 3rem;">
                Explore our stunning collection of events we've organized. Each photo and video tells a story 
                of joy, celebration, and unforgettable moments we've helped create.
            </p>
        </div>
    </div>
</section>

<!-- Gallery Filters -->
<section class="section" style="background: var(--bg-secondary); padding: 2rem 0;">
    <div class="container">
        <div class="filters-container" style="position: relative;">
            <!-- Navigation arrows -->
            <button class="filter-arrow filter-arrow-left" id="galleryFilterArrowLeft" style="display: none;" aria-label="Scroll left">
                <i class="fas fa-chevron-left"></i>
            </button>
            <button class="filter-arrow filter-arrow-right" id="galleryFilterArrowRight" style="display: none;" aria-label="Scroll right">
                <i class="fas fa-chevron-right"></i>
            </button>
            
            <div class="gallery-filters">
                <button class="filter-btn active" data-filter="all">All Events</button>
                <?php if (!empty($categories)): ?>
                    <?php foreach ($categories as $category): ?>
                        <button class="filter-btn" data-filter="<?php echo strtolower($category); ?>">
                            <?php echo htmlspecialchars($category); ?>
                        </button>
                    <?php endforeach; ?>
                <?php else: ?>
                    <!-- Default categories -->
                    <button class="filter-btn" data-filter="wedding">Weddings</button>
                    <button class="filter-btn" data-filter="engagement">Engagements</button>
                    <button class="filter-btn" data-filter="reception">Receptions</button>
                    <button class="filter-btn" data-filter="birthday">Birthdays</button>
                    <button class="filter-btn" data-filter="corporate">Corporate</button>
                    <button class="filter-btn" data-filter="anniversary">Anniversaries</button>
                <?php endif; ?>
            </div>
        </div>
    </div>
</section>

<!-- Gallery Grid -->
<section class="section">
    <div class="container">
        <div class="gallery-grid">
            <?php if (empty($gallery_items)): ?>
                <!-- Default gallery items if no data from database -->
                <div class="gallery-item animate-fade-up" data-category="wedding" data-caption="Beautiful Wedding Ceremony">
                    <img src="assets/images/gallery/wedding-1.jpg" alt="Beautiful wedding decoration" 
                         data-src="assets/images/gallery/wedding-1.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Royal Wedding</h4>
                            <p>Elegant mandap decoration</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="engagement" data-caption="Romantic Engagement Setup">
                    <img src="assets/images/gallery/engagement-1.jpg" alt="Elegant engagement setup" 
                         data-src="assets/images/gallery/engagement-1.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Ring Ceremony</h4>
                            <p>Intimate engagement celebration</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="reception" data-caption="Grand Reception Party">
                    <img src="assets/images/gallery/reception-1.jpg" alt="Grand reception party" 
                         data-src="assets/images/gallery/reception-1.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Reception Party</h4>
                            <p>Luxurious celebration setup</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="birthday" data-caption="Fun Birthday Celebration">
                    <img src="assets/images/gallery/birthday-1.jpg" alt="Fun birthday celebration" 
                         data-src="assets/images/gallery/birthday-1.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Birthday Party</h4>
                            <p>Themed decoration and fun</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="corporate" data-caption="Professional Corporate Event">
                    <img src="assets/images/gallery/corporate-1.jpg" alt="Professional corporate event" 
                         data-src="assets/images/gallery/corporate-1.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Corporate Event</h4>
                            <p>Professional conference setup</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="wedding" data-caption="Wedding Ceremony Decoration">
                    <img src="assets/images/gallery/wedding-2.jpg" alt="Wedding ceremony decoration" 
                         data-src="assets/images/gallery/wedding-2.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Wedding Decor</h4>
                            <p>Traditional ceremony setup</p>
                        </div>
                    </div>
                </div>

                <!-- More default items -->
                <div class="gallery-item animate-fade-up" data-category="wedding" data-caption="Bridal Makeup & Styling">
                    <img src="assets/images/gallery/bridal-1.jpg" alt="Bridal makeup and styling" 
                         data-src="assets/images/gallery/bridal-1.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Bridal Beauty</h4>
                            <p>Professional makeup & styling</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="reception" data-caption="Dance Floor Setup">
                    <img src="assets/images/gallery/dance-floor.jpg" alt="Dance floor setup" 
                         data-src="assets/images/gallery/dance-floor.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Dance Floor</h4>
                            <p>Entertainment and music</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="birthday" data-caption="Kids Birthday Theme">
                    <img src="assets/images/gallery/kids-birthday.jpg" alt="Kids birthday party" 
                         data-src="assets/images/gallery/kids-birthday.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Kids Party</h4>
                            <p>Superhero themed celebration</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="corporate" data-caption="Team Building Event">
                    <img src="assets/images/gallery/team-building.jpg" alt="Team building event" 
                         data-src="assets/images/gallery/team-building.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Team Building</h4>
                            <p>Corporate team activities</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="anniversary" data-caption="Anniversary Celebration">
                    <img src="assets/images/gallery/anniversary-1.jpg" alt="Anniversary celebration" 
                         data-src="assets/images/gallery/anniversary-1.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Anniversary</h4>
                            <p>Romantic milestone celebration</p>
                        </div>
                    </div>
                </div>

                <div class="gallery-item animate-fade-up" data-category="engagement" data-caption="Engagement Photography">
                    <img src="assets/images/gallery/engagement-photo.jpg" alt="Engagement photography" 
                         data-src="assets/images/gallery/engagement-photo.jpg" class="lazy">
                    <div class="gallery-overlay">
                        <i class="fas fa-search-plus"></i>
                        <div class="gallery-info">
                            <h4>Couple Photography</h4>
                            <p>Beautiful engagement portraits</p>
                        </div>
                    </div>
                </div>

            <?php else: ?>
                <!-- Dynamic gallery items from database -->
                <?php foreach ($gallery_items as $item): ?>
                <div class="gallery-item animate-fade-up" 
                     data-category="<?php echo strtolower($item['service_category']); ?>"
                     data-caption="<?php echo htmlspecialchars($item['title'] ?: $item['description']); ?>">
                    <?php if ($item['file_type'] === 'image'): ?>
                    <img src="../uploads/gallery/<?php echo htmlspecialchars($item['file_name']); ?>" 
                         alt="<?php echo htmlspecialchars($item['alt_text'] ?: $item['title']); ?>"
                         data-src="../uploads/gallery/<?php echo htmlspecialchars($item['file_name']); ?>" 
                         class="lazy">
                    <?php else: ?>
                    <video src="../uploads/gallery/<?php echo htmlspecialchars($item['file_name']); ?>" 
                           title="<?php echo htmlspecialchars($item['title']); ?>" muted>
                    </video>
                    <?php endif; ?>
                    <div class="gallery-overlay">
                        <i class="fas fa-<?php echo $item['file_type'] === 'video' ? 'play' : 'search-plus'; ?>"></i>
                        <?php if ($item['title'] || $item['description']): ?>
                        <div class="gallery-info">
                            <?php if ($item['title']): ?>
                            <h4><?php echo htmlspecialchars($item['title']); ?></h4>
                            <?php endif; ?>
                            <?php if ($item['description']): ?>
                            <p><?php echo htmlspecialchars($item['description']); ?></p>
                            <?php endif; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- Load More Button -->
        <div class="text-center" style="margin-top: 3rem;">
            <button id="loadMoreBtn" class="btn btn-outline btn-lg" style="display: none;">
                <i class="fas fa-images"></i>
                Load More Images
            </button>
        </div>
    </div>
</section>

<!-- Gallery Statistics -->
<section class="section" style="background: linear-gradient(135deg, #F9FAFB 0%, #F3F4F6 100%);">
    <div class="container">
        <div class="grid grid-4">
            <div class="stat-card text-center animate-fade-up">
                <div class="stat-number">500+</div>
                <div class="stat-label">Events Organized</div>
            </div>
            <div class="stat-card text-center animate-fade-up">
                <div class="stat-number">2000+</div>
                <div class="stat-label">Happy Photos</div>
            </div>
            <div class="stat-card text-center animate-fade-up">
                <div class="stat-number">100+</div>
                <div class="stat-label">Wedding Videos</div>
            </div>
            <div class="stat-card text-center animate-fade-up">
                <div class="stat-number">4.9★</div>
                <div class="stat-label">Client Rating</div>
            </div>
        </div>
    </div>
</section>

<!-- Call to Action -->
<section class="section" style="background: var(--bg-gradient); color: white;">
    <div class="container">
        <div class="text-center animate-fade-up">
            <h2 class="section-title" style="color: white;">Love What You See?</h2>
            <p style="font-size: 1.2rem; margin-bottom: 2rem; color: rgba(255,255,255,0.9);">
                Let us create beautiful memories for your special event too. Contact us today for a personalized consultation.
            </p>
            <div style="display: flex; justify-content: center; gap: 1rem; flex-wrap: wrap;">
                <a href="contact.php" class="btn btn-secondary btn-lg">
                    <i class="fas fa-envelope"></i>
                    Plan Your Event
                </a>
                <a href="services.php" class="btn btn-outline btn-lg" style="border-color: white; color: white;">
                    <i class="fas fa-concierge-bell"></i>
                    View Services
                </a>
            </div>
        </div>
    </div>
</section>

<style>
/* Gallery page specific styles */
.text-center { text-align: center; }


.gallery-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.gallery-item {
    position: relative;
    aspect-ratio: 4/3;
    border-radius: var(--radius-lg);
    overflow: hidden;
    cursor: pointer;
    box-shadow: var(--shadow-md);
    transition: var(--transition-normal);
    background: var(--glass-white);
    min-height: 200px; /* Fallback for older browsers */
}

.gallery-item:hover {
    transform: scale(1.02);
    box-shadow: var(--shadow-xl);
}

.gallery-item img,
.gallery-item video {
    width: 100%;
    height: 100%;
    object-fit: cover;
    object-position: center;
    transition: var(--transition-slow);
    display: block;
}

.gallery-item:hover img,
.gallery-item:hover video {
    transform: scale(1.1);
}

.gallery-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: linear-gradient(135deg, rgba(107, 70, 193, 0.8), rgba(124, 45, 18, 0.8));
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    opacity: 0;
    transition: var(--transition-normal);
    text-align: center;
    color: white;
    padding: 1rem;
}

.gallery-item:hover .gallery-overlay {
    opacity: 1;
}

.gallery-overlay i {
    font-size: 2rem;
    margin-bottom: 1rem;
    color: var(--accent-gold);
}

.gallery-info h4 {
    font-family: var(--font-heading);
    font-size: 1.2rem;
    margin-bottom: 0.5rem;
    color: white;
}

.gallery-info p {
    font-size: 0.9rem;
    opacity: 0.9;
    margin: 0;
}

.stat-card {
    padding: 2rem;
    background: var(--glass-white);
    backdrop-filter: var(--backdrop-blur);
    border-radius: var(--radius-lg);
    border: 1px solid var(--glass-border);
    transition: var(--transition-normal);
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: var(--shadow-lg);
}

.stat-number {
    font-size: 2.5rem;
    font-weight: 700;
    color: var(--primary-purple);
    margin-bottom: 0.5rem;
    font-family: var(--font-heading);
}

.stat-label {
    font-size: 1rem;
    color: var(--text-secondary);
    font-weight: 500;
}

/* Gallery item hidden state for filtering */
.gallery-item.hidden {
    opacity: 0;
    transform: scale(0.9);
    transition: opacity 0.3s ease, transform 0.3s ease;
    pointer-events: none;
}

.gallery-item:not(.hidden) {
    opacity: 1;
    transform: scale(1);
    transition: opacity 0.3s ease, transform 0.3s ease;
}

/* Loading animation */
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

.gallery-item.fade-in {
    animation: fadeIn 0.5s ease-out;
}

/* Image loading state */
.gallery-item img {
    opacity: 1;
    transition: opacity 0.3s ease;
}

@media (max-width: 768px) {
    .gallery-grid {
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 1.2rem;
    }
    
    
    .stat-number {
        font-size: 2rem;
    }
    
    .gallery-overlay {
        padding: 0.5rem;
    }
    
    .gallery-overlay i {
        font-size: 1.5rem;
        margin-bottom: 0.5rem;
    }
    
    .gallery-info h4 {
        font-size: 1rem;
    }
    
    .gallery-info p {
        font-size: 0.8rem;
    }
}

@media (max-width: 480px) {
    .gallery-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .gallery-item {
        min-height: 180px; /* Smaller minimum height on mobile */
    }
    
    .stat-card {
        padding: 1.5rem 1rem;
    }
}
</style>

<?php 
// Add page-specific JavaScript initialization
$page_js_init = "
    // Initialize gallery filtering
    if (typeof initializeGalleryFilters === 'function') {
        initializeGalleryFilters();
    }
    
    // Initialize gallery filter arrows
    if (typeof initializeFilterArrows === 'function') {
        initializeFilterArrows('gallery');
    }
    
    // Simple image error handling
    document.querySelectorAll('.gallery-item img').forEach(img => {
        img.addEventListener('error', function() {
            this.style.background = '#f3f4f6';
            this.style.display = 'flex';
            this.style.alignItems = 'center';
            this.style.justifyContent = 'center';
            this.innerHTML = '🖼️';
        });
    });
    
    // Add load more functionality if needed
    const loadMoreBtn = document.getElementById('loadMoreBtn');
    if (loadMoreBtn) {
        // This would be implemented based on pagination needs
        loadMoreBtn.style.display = 'none'; // Hide for now
    }
";

include '../includes/footer.php'; 
?>